% Code for: Capital Reallocation and Aggregate Productivity
% R. Cooper & I. Schott
% Code version: September 2022
function res        = solve_agg(PARMS,steady,agg_shock,upd_graphs)

if nargin == 3
    upd_graphs = 1;
end

if strcmp(agg_shock,'A')
    % Set shocks to F to zero
    PARMS.sigF      = 0;
    PARMS.rhoF      = 0;
elseif strcmp(agg_shock,'F')
    PARMS.sigA      = 0;
    PARMS.rhoA      = 0;
end

% Data
define_data_moms_agg

%% Set up simulation of aggregate shock

% Initial values from steady state
mu_a            = steady.mu_a;
mu_n            = steady.mu_n;
phi_a           = steady.phi_a;
phi_n           = steady.phi_n;
k_a             = steady.k_a;
pistar          = steady.pistar;
cstar           = steady.cstar;

%% Simulation of T periods to estimate aggregate moments
% Fix random seed for replicability
rng('default')
rng(1)
T               = 200;
% A shocks
sig             = PARMS.sigA*sqrt(1-PARMS.rhoA^2); % scales the innovations
e               = randn(T,1) * sig; % Sequence of random innovations
logA            = zeros(T,1);
for tt = 2:T
    logA(tt)    = PARMS.rhoA * logA(tt-1) + e(tt);
end
At              = exp(logA); %#ok<*NASGU>
clear logA

% F shocks
sig             = PARMS.sigF*sqrt(1-PARMS.rhoF^2); % scales the innovations
e               = randn(T,1) * sig; % innovation
logF            = zeros(T,1);
for tt = 2:T
    logF(tt)    = PARMS.rhoF * logF(tt-1) + e(tt);
end
Ft              = exp(logF) * PARMS.F_upper;
clear logF

% Initialize
pit             = zeros(T,1);
ct              = zeros(T,1);
mu_at           = ones(T,1) * mu_a;
phi_at          = ones(T,1) * phi_a;
mu_nt           = ones(T,1) * mu_n;
phi_nt          = ones(T,1) * phi_n;
kt              = ones(T,PARMS.N).*k_a'; % initial guess

% Run simulation
run_sim

% Compute additional output
y_at    = PARMS.meaneps*mu_at + phi_at; % Net of aggregate productivity
y_nt    = PARMS.meaneps*mu_nt + phi_nt; % Net of aggregate productivity
ACt     = pit.^2 / 2 .* Ft;
Yt      = At .* (pit.*y_at + (1-pit).*y_nt ) ;
SRt     = Yt; % Solow residual

% Compute reallocation and dispersion
compute_R_sigma

%% Compute model moments
compute_agg_moments

tmp_data    = cat(1,datamoms{:,1});
tmp_model   = cat(1,simmoms{:,1});

selection   = [1  2]; %
res         = mean(abs((tmp_data(selection)-tmp_model(selection))./tmp_data(selection)));
fprintf('\n The distance between data and targeted model moments is %4.3f%%\n', 100*res)

%% Collect moments for tables and figures

if upd_graphs == 1

    Table_3_create_modeloutput

    Figure_6_create_modeloutput

    Figure_4_5_create_modeloutput

end
